;***********************************************
;*         V2MPMLDR.ASM                        *
;*         FLOPPY CBIOS FOR CP/M 2.XX          *
;*         MADE INTO MP/M LOADER BIOS          *
;*         BRUCE JONES 12/15/2003              *
;***********************************************
; WORKS WITH VF-II



CPVERS	==	22H	;CP/M VERSION NUMBER
BIVERS	==	0H	;BIOS VERSION NUMBER


	.Z80			
	.PABS		
	.PHEX
	.XLINK


VDBCON	=\	\INPUT 0 FOR VDB-8024 CONSOLE, 1 IF SBC-200 USART  \


TRUE	==	0FFFFH	;TRUE AND FALSE VALUES
FALSE	==	#TRUE

Z3S	==	TRUE	;THIS VERSION FOR DMA OR PIO BOARD


;****************************************************
;        ASK FOR CP/M SYSTEM SIZE ETC. NEXT
;****************************************************


NODSK	==	2
STEP	==	0
STEPS   ==	0
DLY	==	10
CPMDSK	==	0004H	;CURRENT DISK NUMBER ADDR
IOBYTE	==	0003H	;INTEL I/O BYTE ADDR

;MEMORY AVAILABLE FUNCTION (* 1024 BYTES)  NEXT

MEMFCT	==	3
STEPR	=	STEP

CR	==	0DH
LF	==	0AH
BELL	==	07H
CLEAR	==	1AH


;*****************************************************
;*      SYSTEM DATA I/O PORT CONSTANTS NEXT
;*****************************************************

;                KEYBOARD & SCREEN

TTYI	==	1	;TTY INPUT PORT ADDRESS
TTYO	==	1	;TTY OUTPUT PORT ADDRESS
TTYS	==	0	;TTY I/O STATUS PORT ADDRESS
TTYDA	==	2	;TTY DATA AVAILABLE MASK
TTYBE	==	4	;TTY XMIT BUFFER EMPTY MASK

;                   MODEM PORT

CRTI	==	5	;CRT INPUT PORT ADDRESS (MODEM)
CRTO	==	5	;CRT OUTPUT PORT ADDRESS (MODEM)
CRTS	==	4	;CRT I/O STATUS PORT ADDRESS
CRTDA	==	1	;CRT DATA AVAILABLE MASK
CRTBE	==	80H	;CRT XMIT BUFFER EMPTY MASK



;    PRINTER PORTS FOLLOW , PARALLEL PRINTER			

LSTST	==	6	;LIST DEVICE STATUS PORT
LSTDAT	==	7	;LIST DEVICE DATA PORT
CEN	==	9	;TURN ON CENTRONICS PORT
SPIN	==	8	;TURN ON SPINWRITER PORT


;          SERIAL PRINTER PORT NEXT

SERST	==	2	;SERIAL PRINTER STATUS
SERDT	==	3	;SERIAL PRINTER DATA PORT

		
CMSK	==	00000011B ;IOBYTE MASK FOR CONSOLE
LMSK	==	11000000B ;IOBYTE MASK FOR LIST


;****************************************************
;             FLOPPY DISK EQUATES.
;****************************************************


;****************************************************
;            PIO CONTROLLER PORTS
;****************************************************

DCMD	==	63H	;Z3S COMMAND/CONTROL PORT
W	==	64H	;WDC 179X ADDRESS
WCMD	==	W+0	;COMMAND PORT
WSTAT	==	W+0	;STATUS PORT
WTRACK	==	W+1	;TRACK REG
WSECT	==	W+2	;SECTOR REG
WDATA	==	W+3	;DATA I/O REG

;****************************************************
;        COMMON 179X CONTROLLER COMMANDS
;****************************************************

WHOME	==	00001000B ;HOME COMMAND
WREAD	==	10001000B ;READ SECTOR COMMAND
WWRITE	==	10101000B ;WRITE SECTOR COMMAND
WSEEK	==	00011000B ;SEEK TO GIVEN TRACK COMMAND
WUNLD	==	00010000B ;SEEK AND UNLOAD HEAD COMMAND
WLOAD	==	00011000B ;SEEK AND LOAD HEAD COMMAND


;****************************************************
;          COMMON 179X CONTROLLER STATUS
;****************************************************

WBBUSY	==	0	;179X BUSY STATUS BIT
WBSID1	==	1	;SIDE SELECT FLAG COMMAND BIT
WBDEL	==	2	;HEAD SETTLE DELAY COMMAND BIT
WBWRIT	==	5	;READ/WRITE DISTINGUISHING BIT
WBRNF	==	4	;RECORD NOT FOUND STATUS BIT
WSREAD	==	10011100B ;READ SECTOR STATUS MASK
WSWRIT	==	11111100B ;WRITE SECTOR STATUS MASK
WSSEEK	==	10011000B ;SEEK STATUS MASK
WFCINT	==	11010000B ;FORCE INTERRUPT COMMAND


;*****************************************************
;*          BIOS  DISK I/O ERROR TYPES
;*****************************************************

ERR1	==	1	;   WARM BOOT
ERR2	==	2	;   READING DDB
ERR3	==	3	;   BUFFERED READ
ERR4	==	4	;   UNBUFFERED READ
ERR5	==	5	;   UNBUFFERED WRITE
ERR6	==	6	;   PREREAD FOR BUFFERED WRITE
ERR7	==	7	;   BUFFERED WRITE
ERR8	==	8	;   PHYSICAL I/O (SOFT ERROR)

;******************************************************
; PRIMARY JUMP TABLE. ALL CALLS FROM CP/M TO THE CBIOS
; COME THROUGH THIS TABLE.

; CONSOLE AND LIST CALLS BRANCH TO ROUTINES WHICH USE
; THE IOBYTE TO SELECT THE ACTUAL I/O DEVICE. IN ORDER
; TO PRESERVE THE CAPABILITY OF CHANGING PHYSICAL I/O
; DEVICES UNDER CP/M CONTROL, THE PHYSICAL I/O JUMPS IN
; THIS TABLE SHOULD NOT BE ALTERED.

; IF THE CONSOLE OUTPUT ROUTINE IS CHANGED, THE NEW
; ROUTINE MUST CONFORM TO ZAPPLE CONVENTIONS. IE, NO
; REGISTERS MUST BE DISTURBED, EXCEPT THAT THE BYTE
; WHICH WAS OUTPUT MUST BE RETURNED IN A.

; THE JUMP FOR THE COLD BOOT MUST NOT BE DISTURBED,
; SINCE THE FIRST INSTRUCTION IN THE COLD BOOT POINTS
; TO XTABLE, THE EXTERNAL BIOS DISK TABLE.
;******************************************************

	.LOC	1700H



	JMP	BOOT	;COLD BOOT
WBOOTE:	JMP	WBOOT	;WARM BOOT
	JMP	CONST	;CONSOLE STATUS
	JMP	CONIN	;CONSOLE CHARACTER IN
BIOOUT:	JMP	CONOUT	;CONSOLE CHARACTER OUT,
			;   ALSO USED BY THE CBIOS
	JMP	LIST	;LIST CHARACTER OUT
	JMP	PUNCH	;PUNCH CHARACTER OUT
	JMP	READER	;READER CHARACTER IN
	JMP	HOME	;MOVE HEAD TO HOME POSITION
	JMP	SELDSK	;SELECT DISK
	JMP	SETTRK	;SET TRACK NUMBER
	JMP	SETSEC	;SET SECTOR NUMBER
	JMP	SETDMA	;SET DMA ADDRESS
	JMP	READ	;READ DISK
	JMP	WRITE	;WRITE DISK
	JMP	LISTST	;RETURN LIST STATUS
	JMP	SECTRA	;SECTOR TRANSLATE



;******************************************************
; STEP SPEED TABLE. THIS TABLE TELLS THE BIOS WHAT SIZE
; OF DRIVE IS AT EACH ADDRESS AND WHAT TRACK TO TRACK
; STEPPING SPEED TO USE FOR THAT DRIVE. THE BITS MEAN
; THE FOLLOWING:
;******************************************************

;	1000 0011
;	^      ^
;	:      :- THE 179X STEP SPEED BITS. THESE ARE:
;	:		VALUE	8"	5"
;	:		00	3 MS	6 MS
;	:		01	6 MS	12 MS
;	:		10	10 MS	20 MS
;	:		11	15 MS	30 MS
;	:
;	:-------- 0=8 INCH DISK AT THIS ADDRESS,
;		  1=5 INCH DISK AT THIS ADDRESS.

SPDTAB: .BYTE   STEPS   ;DRIVE A
        .BYTE   STEPS   ;DRIVE B
	.BYTE	STEPS	;DRIVE C
	.BYTE	STEPS	;DRIVE D


;******************************************************
; THIS TABLE PROVIDES INFORMATION TO EXTERNAL PROGRAMS.
; IT IS USED BY SEVERAL CBIOS DEPENDENT UTILITIES.
; THE UTILITIES LOCATE THIS TABLE BY USING THE LXI
; INSTRUCTION AT THE BEGINNING OF THE COLD BOOT. THE
; TABLE ALSO CONTAINS SEVERAL BYTES WHICH CAN BE
; CHANGED BY THE USER TO CUSTOMIZE HIS CBIOS.

; CHANGES MUST BE MADE WITH CAUTION!
;*****************************************************

XTABLE:	.BYTE	0FDH,0DDH ;IDENTIFICATION BYTES
	.BYTE	0	;COMPATIBILITY BYTE
	.BYTE	0H	;FLAG BYTE FOR Z3S CBIOS
	.BYTE	0	;FLAG BYTE
	.BYTE	CPVERS	;VERSION IDENTIFICATION
	.BYTE	BIVERS

; OK TO CHANGE THE NEXT TWO BYTES.
INITIO:	.BYTE	01010100B ;INITIAL I/O BYTE AND
	.BYTE	0H	;INITIAL DISK AFTER COLD BOOT


;****************************************************
; FMICRO & FMACRO SET TO GIVE 3 USER TRIES AT
; CORRECTING PROBLEM ON DISK
;****************************************************

FRETRY:
FMICRO:	.BYTE	20	;FLOPPY MICRO RETRY LIMIT
FMACRO:	.BYTE	126	;FLOPPY MACRO RETRY LIMIT
FSOFT:	.BYTE	FALSE	;RESERVED FOR INTERNAL USE
			;WHEN SET TRUE SOFT ERRORS
			;ARE DISPLAYED ON SCREEN
	.WORD	DSKRST	;DISK RESET ENTRY ADDRESS
	.WORD	HEADLD	;HEAD LOAD ENTRY ADDRESS

;******************************************************
; THE SYSTEM CONFIGURATION OPTION BYTES FOLLOW
; THESE BYTES DETERMINE:
; LETTER PRINTER SHIFT SUPPRESSION
; INITIAL ERROR MESSAGE PRINT SUPPRESSION
; NUMBER OF DRIVES IN SYSTEM
; DEFAULT LIST DEVICE
;*****************************************************

CFGOPT:

	CFGTAB:	.BYTE	'C'	;SHOW IT HAS A CONFIGURATION
				;TABLE
	
	SFTSUP:	.BYTE	0	;SHIFT CHARS TO SPINWRITER
				; 0 = NOT SUPPRESSED
				
	ERRSUP:	.BYTE	0	;INITIAL DISK ERROR MESSAGE
				;0 = NOT SUPPRESSED

	ACTDSK:	.BYTE	NODSK	;NUMBER OF DISKS SUPPORTED
				;NORMALLY 2 DRIVES

	LTYPE:	.BYTE	'L'	;DEFAULT LIST DEVICE
				;NORMALLY THE LINE PRINTER		


;******************************************************
; Z3S OPERATING CONTROL TABLE. THIS TABLE CONTAINS
; THOSE BITS IN THE Z3S CONTROL BYTE WHICH SELECT THE
; OPERATING MODE. THESE BITS ARE COMBINED WITH ADDRESS
; CONTROL BITS TO FORM A COMPLETE Z3S CONTROL BYTE. THE
; BITS ARE DEFINED AS FOLLOWS:
;******************************************************
; VF-II VERSION OF TABLE
;
;	1111 0000
;	^^^^ ^
;	:::: +- THESE BITS IN ADDRESS CONTROL TABLE
;	::::
;	:::+--- 0=SIDE 1, 1=SIDE 0
;       ::+---- 1=8 INCH DISK, 0=5 INCH DISK
;	:+----- 1=SINGLE DENSITY (FM), 0=DOUBLE (MFM)
;	+------ 0=ENABLE HARDWARE WAIT, 1=DISABLE

OPRTAB:

; FOR VERSAFLOPPY II
; SIZE &	 
; DENSITY	 

C8S:	.BYTE	11110000B	;SIDE 0, NO WAIT
	.BYTE	01110000B	;SIDE 0, WAIT
	.BYTE	11100000B	;SIDE 1, NO WAIT
	.BYTE	01100000B	;SIDE 1, WAIT

C5S:	.BYTE	11010000B
	.BYTE	01010000B
	.BYTE	11000000B
	.BYTE	01000000B

C8D:	.BYTE	10110000B	;SIDE 0, NO WAIT
	.BYTE	00110000B	;SIDE 0, WAIT
	.BYTE	10100000B	;SIDE 1, NO WAIT
	.BYTE	00100000B	;SIDE 1, WAIT

C5D:	.BYTE	10010000B
	.BYTE	00010000B
	.BYTE	10000000B
	.BYTE	00000000B


;*****************************************************
; Z3S ADDRESS CONTROL TABLE. THIS TABLE CONTAINS THSOE
; BITS IN THE Z3S BYTE WHICH SELECT THE ACTIVE DISK.
; THESE BITS ARE COMBINED WITH OPERATING CONTROL BITS
; TO FORM A COMPLETE Z3S CONTROL BYTE.
;*****************************************************

ADRTAB:	.BYTE	1110B	;DRIVE A
	.BYTE	1101B	;DRIVE B
	.BYTE	1011B	;DRIVE C
	.BYTE	0111B	;DRIVE D


;****************************************************
; INTERRUPT DISABLE AND ENABLE. DISINT IS CALLED BY
; THE FLOPPY DISK PHYSICAL I/O ROUTINE TO DISABLE
; INTERRUPTS BEFORE READING OR WRITING A SECTOR.
; ENAINT IS CALLED TO ENABLE INTERRUPTS AFTER
; COMPLETION OF THE I/O OPERATION.
; IF DMA CONTROLLER NO DI OR EI IS PERFORMED
;****************************************************

DISINT:
	DI
	RET	


;****************************************************
; COLD BOOT ENTRY POINT. THE FIRST FOUR INSTRUCTIONS
; SHOULD NOT BE CHANGED.
;****************************************************

BOOT:
	MVI	A,0
	STA	FIRSTS
	XRA	A
	RET

WBOOT:
	XRA	A
	RET



;****************************************************
; RESET DISK SYSTEM. INVALIDATE CERTAIN FLOPPY DISK
; TABLES AND BYTES TO ALLOW CHANGING DISKS. CALLED BY
; COLD BOOT, WARM BOOT, AND SOME EXTERNAL ROUTINES.
;****************************************************

DSKRST:
	XRA	A
	LXI	H,UNACNT ;INVALIDATE UNALLOCATED COUNT
	MVI	B,UNALEN
..LOP:	MOV	M,A
	INX	H
	DJNZ	..LOP
	LXI	H,ACTDSK ;POINT TO # DISKS	
	MOV	B,M	 ;GET NUMBER INTO B
	LXI	D,APBDIS ;GET DISTANCE BETWEEN FD APBS
	LXI	H,APB0+(FLAG-ATABLE) ;POINT TO FLAG
..LOOP:	MOV	M,A	;INVALIDATE ALL FLOPPY DISK
	DAD	D	;   APBS BY CLEARING FLAGS
	DJNZ	..LOOP

	DCR	A
	STA	OLDFLO	;FORCE HEAD UNLOAD/LOAD
	STA	ADISK	;INVALIDATE ATABLE
	RET



;****************************************************
; LOAD HEAD ON CURRENT DISK. NOT YET IMPLEMENTED.
;****************************************************

HEADLD:	RET



;******************************************************
; LOGICALLY SELECT THE DISK DRIVE FOR FUTURE READS AND
; WRITES TO THAT PASSED IN REGISTER C. IF THE DDB FOR
; THE DRIVE HAS NOT YET BEEN READ, THEN READ IT IN FROM
; THE DISK. OTHERWISE, DON'T PERFORM A PHYSICAL SELECT
; UNTIL A READ OR WRITE SECTOR CALL IS MADE. NOTE THAT
; THE DPH, APB, DPB, AND TRANSLATE TABLE FOR THE DRIVE
; ARE ALL VALID AT THE COMPLETION OF THIS CALL.
;******************************************************

SELDSK:
	DI
	LDA	FIRSTS
	CPI	0
	JZ	BCOMN
	PUSH	B
	PUSH	D
	CALL	BOOT
	POP	D
	POP	B

BCOMN:	CALL	DSKRST	;RESET DISK SYSTEM

	MOV	A,C
	LXI	H,ACTDSK;POINT TO ACTUAL DISK #
	CMP	M	;IN RANGE(y/n)
	JNC	..BAD	;NO
	STA	SEKDSK	;D.R. HOST DISK
	STA	PDISK
	MOV	L,C	;GET APB, DPH ADDRESSES
	CALL	GETDPH	;GET RAM LOCATION FOR DPH
	SDED	APBADR	;SAVE APB ADDRESS
	SHLD	DPHADR	;SAVE DPH ADDRESS
	CALL	GETAPB	;GET ATABLE FOR THIS DISK
	LDA	FLAG
	ORA	A	;DDB PROCESSED(y/n)
	JNZ	..OK	;YES


	XRA	A	;WE WILL READ FROM TRACK 0
	STA	PTRACK
	MVI	A,8	;AND SECTOR 8
	STA	PSECT
	LXI	H,RDBUFF ;INTO THE READ BUFFER
	SHLD	PDMA

	LDA	ADISK	;GET DISK NUMBER
	LXI	H,SPDTAB ;POINT TO STEP SPEED TABLE
	MOV	E,A
	MVI	D,0
	DAD	D
	MOV	A,M	;GET SPEED BYTE FOR THIS DISK
	MOV	B,A
	ANI	3	;ISOLATE SPEED BITS
	MOV	H,A
	MVI	L,0FFH	;CURRENT TRACK - UNKNOWN
	SHLD	TRACK	;UPDATE ATABLE TRACK AND SPEED


;****************************************************
;          ASSUME 8" S.S.S.D. DISK NEXT
;****************************************************

	MVI	A,00000001B ;INITIAL FLAG FOR 8" DRIVE
	LXI	H,STDDDB ;STANDARD 8" DDB ADDRESS
	BIT	7,B	;8" DISK(y/n)
..EIG:	STA	FLAG
	SHLD	SAVADR

	LXI	H,1	;AT LEAST 1 SYSTEM TRACK
	SHLD	OFF
	LXI	H,2+32<8 ;512 BYTE PHYSICAL SECTOR LEN,
	SHLD	SSLEN	;   AT LEAST 32 LOGICAL SECTORS

	CALL	GETD3S	;GET Z3S CTRL BYTES INTO ATABLE
	CALL	PREAD	;GET THE DDB
	JRNC	..YUP	;WE GOT SOMETHING


;****************************************************
; IF ERROR IS R.N.F. THEN SHOW DISK READ FAILED
; ELSE TEST IT FOR GOOD DISK TYPE
;****************************************************

	BIT	WBRNF,A	;RNF ERROR(y/n)
	JRZ	..ERR	;NOPE, GIVE UP


;****************************************************
;    SET UP FOR STANDARD 8" S.S.S.D.  DISK HERE
;****************************************************

..NOV:

	LHLD	SAVADR	;POINT TO STANDARD DDB
	LXI	B,128-10
	LXI	D,RDBUFF+384+10
	LDIR		;FIX RD BUFFER TO BE A STD DDB
	JMPR	..COMP

..YUP:	LHLD	RDBUFF+384 ;GET VALIDITY BYTES FROM DDB
	LXI	D,0DDH+0FDH<8 ;EXPECTED VALUE OF BYTES
	ORA	A
	DSBC	D	;DDB VALID(y/n)
	JRNZ	..NOV	;NOPE
	LHLD	RDBUFF+384+2 ;MORE VALIDITY BYTES
	DSBC	D	;DDB VALID(y/n)
	JRNZ	..NOV	;NOPE

	LDA	RDBUFF+384+4 ;TEST FOR COMPATIBILITY
	ANI	11111110B
	JRNZ	..BAD	;GIVE UP ON THIS DISK

..COMP:	CALL	PUTAPB	;UPDATE TRACK, SPEED IN APB

	LHLD	APBADR	;GET APB ADDR FOR THIS DISK
	LXI	D,FLAG-ATABLE ;POINT TO WHERE FLAG GOES
	DAD	D
	XCHG		;MAKE THIS THE DESTINATION ADDR
	LXI	H,RDBUFF+384+10 ;FROM FLAG IN RD BUFFER
	LXI	B,ALEN-(FLAG-ATABLE)+DPBLEN+TRALEN
	LDIR		;MOVE DDB, DPB, TRANS INTO APB


;***************************************************
;*           NOW SET UP ALLOCATION SIZE
;*           FOR THE DISK JUST SELECTED
;* GET THE ALLOCATION SIZE FROM THE DSM VALUE IN
;* THE DPB
;***************************************************

	LDA	RDBUFF+384+19	;GET DSM
	MOV	C,A	;SAVE VALUE
	LDA	SEKDSK
	MOV	E,A
	MVI	D,0
	LXI	H,ALOCSZ
	DAD	D
	MOV	M,C	;SAVE FOR THIS DISK

	MVI	A,0FFH	;UPDATE ATABLE FROM APB
	STA	ADISK
	CALL	GETAPB
	CALL	GETD3S	;PUT VALID Z3S BYTES IN ATABLE
	CALL	PUTAPB	;UPDATE APB FROM
			;   FULLY VALID ATABLE

..OK:	LHLD	DPHADR	;RETURN DPH ADDRESS
	RET

..ERR:	MVI	C,ERR2
	CALL	EPRINT	;PRINT ERROR IN READING DDB

..BAD:	XRA	A	;DESELECT INVALID DRIVE
	STA	CPMDSK
	MOV	H,A	;ERROR RETURN CODE
	MOV	L,A
	RET



;*******************************************************
; SET TRACK FOR FUTURE READS OR WRITES TO TRACK 0. ALSO
; PARTIALLY RESET THE DISK SYSTEM TO ALLOW FOR CHANGED
; DISKS.

; AFTER A USER PROGRAM CALLS THE BDOS TO RESET THE DISK
; SYSTEM, CP/M CALLS HOME BEFORE ACCESSING THE
; DIRECTORY ON THE NEW DISK. CLEAR THE READ BUFFER
; SINCE IT IS INVALID IF DISKS HAVE BEEN CHANGED. CLOSE
; THE WRITE BUFFER SO THAT IT IS EMPTY WHEN THE
; DIRECTORY IS ACCESSED. IF THE WRITE BUFFER HAPPENS TO
; BE VALID WHEN DISKS ARE CHANGED THE NEW DISK WILL GET
; CLOBBERED. FORTUNATELY, THE WRITE BUFFER WILL NOT BE
; VALID IF ALL FILES ON THE OLD DISK WERE CLOSED BEFORE
; DISKS WERE CHANGED, SINCE THE BUFFER IS CLOSED AFTER
; ALL DIRECTORY WRITES.
;*******************************************************

HOME:

	LDA	HSTWRT	;TEST FOR PENDING WRITE
	ORA	A
	JRNZ	HOMED
	STA	HSTACT	;CLEAR HOST ACTIVE FLAG

HOMED:
	LXI	B,0	;DROP THRU TO SET TRACK TO 0



;****************************************************
; SET TRACK FOR FUTURE READS OR WRITES TO THAT PASSED
; IN REGISTER PAIR BC.
;****************************************************

SETTRK:	SBCD	SEKTRK
	RET



;*****************************************************
; SET SECTOR FOR FUTURE READS OR WRITES TO THAT PASSED
; IN REGISTER PAIR BC.
;*****************************************************

SETSEC:	SBCD	SEKSEC
	RET



;****************************************************
; SET DMA ADDRESS FOR FUTURE READS OR WRITES TO THAT
; PASSED IN REGISTER PAIR BC.
;****************************************************

SETDMA:	SBCD	DMAADD
	RET



;****************************************************
; SECTOR TRANSLATION ROUTINE. THE ROUTINE ONLY
; TRANSLATES SECTORS ON THE USER TRACKS, SINCE CP/M
; ACCESSES THE SYSTEM TRACKS WITHOUT CALLING FOR
; TRANSLATION.
;****************************************************

SECTRA:

..YUP:	XCHG		;HL GETS TRANS TABLE ADDRESS
			;CP/M PASSED IT IN DE

	MOV	A,C	;GET SECTOR #
	SBCD	NEWSEC	;SAVE FOR UNALLOC TEST
	

	DAD	B	;INDEX INTO TABLE, LOGICAL SECTOR
			;IS PASSED IN BC
			
	MOV	L,M	;GET THE TRANSLATED BYTE
	MVI	H,0
	RET


CLOSE:
	LDA	HSTWRT	;SEE IF WE HAVE A PENDING WRITE
	ORA	A
	RZ		;IF NOT RETURN NOW
	JMP	FLUSH	;ELSE FLUSH THE R/W BUFFER


;******************************************************
; CP/M ENTRY POINT FOR SECTOR READS. BUFFERED SECTOR
; READS ARE DONE HERE. BUFFERED READ OPERATIONS REQUIRE
; READING THE SECTOR FROM DISK INTO THE READ BUFFER,
; AND THEN PROVIDING 128 BYTE LOGICAL SECTORS TO THE
; CALLING PROGRAM ON REQUEST.
;******************************************************

READ:

	XRA	A
	STA	UNACNT	;UNACNT=0, WE WON'T WRITE WITHOUT
			;PRE-READS FOR NOW
	INR	A	
	STA	READOP	;SHOW WE ARE DOING A READ OPERATION
	STA	RSFLAG	;MUST READ DATA
	MVI	A,2
	STA	WRTYPE	;TREAT AS UNALLOCATED

	JMP	RWOPER	;DO THE READ


;******************************************************
; CP/M ENTRY POINT FOR SECTOR WRITES. BUFFERED SECTOR
; WRITES ARE DONE HERE. BUFFERED WRITE OPERATIONS
; REQUIRE ACCEPTING 128 BYTE LOGICAL SECTORS FROM THE
; CALLING PROGRAM, ACCUMULATING THEM IN A WRITE BUFFER,
; THEN WRITING THE BUFFER WHEN IT BECOMES FULL. THE
; BUFFER IS IMMEDIATELY WRITTEN OUT IF THE LOGICAL
; SECTOR IS PART OF THE DISK DIRECTORY.
;******************************************************


WRITE:
	XRA	A
	RET


;DO READ OR WRITE OPERATION NEXT

RWOPER:

	CALL	GETTRK	;COMPUTE PHYSICAL TRACK & SECTOR

	LXI	H,HSTACT;GET HOST ACTIVE FLAG
	MOV	A,M	
	MVI	M,1	;SET IT ACTIVE FOR SURE
	ORA	A	;SEE IF IT WAS ACTIVE
	JZ	FILHST	;IF NOT FILL IT
	

;CHECK TO SEE IF SECTOR IN HOST BUFFER TO R/W IS CORRECT
;IF NOT WRITE TO HOST BUFFER IF NEEDED & PREPARE FOR 
;CORRECT HOST BUFFER

;SEE IF DISKS ARE SAME

	LDA	SEKDSK	;COMPARE DISKS FIRST
	LXI	H,HSTDSK
	CMP	M
	JRNZ	NOMAT


;SEE IF TRACKS ARE SAME

	LXI	H,HSTTRK
	LDA	CTRACK
	CMP	M
	JRNZ	NOMAT


;SEE IF SECTORS ARE SAME

	LDA	SEKHST	;COMPARE SECTORS NOW
	LXI	H,HSTSEC
	CMP	M
	JRZ	MATCH


;HOST PARAMETERS DO NOT MATCH CURRENT R/W PARAMETERS
;SEE IF WE HAVE TO FLUSH THE HOST BUFFER

NOMAT:
	LDA	HSTWRT	;SEE IF HOST BUFFER WAS WRITTEN
	ORA	A
	CNZ	FLUSH	;WRITE OUT THE BUFFER IF NOT


;MAY HAVE TO FILL HOST BUFFER
;SEET UP NEW PARAMETERS

FILHST:

	LDA	SEKDSK
	STA	HSTDSK
	LHLD	CTRACK
	SHLD	HSTTRK
	LDA	SEKHST	;THE PHYSICAL SECTOR
	STA	HSTSEC

	LDA	RSFLAG	;SEE IF WE NEED TO READ
	ORA	A
	CNZ	RDHST	;IF SO READ IT
	XRA	A
	STA	HSTWRT	;SHOW NO PENDING WRITE

;WE HAVE CORRECT SECTOR SO COPY DATA TO/FROM DMA BUFFER

MATCH:

	LDA	CREC	;GET RECORD #
	MOV	B,A	;COMPUTE RECORD IN HOST BUFFER
	MVI	C,0
	SRLR	B
	RARR	C
	LXI	H,HSTBUF
	DAD	B	;HL IS NOW HOST BUFFER
	LDED	DMAADD	;DE HAS DMA ADDRESS
	LXI	B,128	;BYTES TO MOVE
	LDA	READOP
	ORA	A	;SEE IF WE ARE READING OR WRITING
	JRNZ	RWMOVE	;SKIP ON READ
	MVI	A,1	;IF A WRITE THEN MARK & COPY TO BUFFER
	STA	HSTWRT	;HSTWRT = 1
	XCHG		;DE IS NOW DESTINATION,= HOST ON WRITE
	LHLD	DMAADD	;HL IS SOURCE, = USER AREA TO GET	

RWMOVE:
	
	LDIR		;MOVE DATA


;NOW CHECK WRITE TYPE FOR DIRECTORY UPDATE

	LDA	WRTYPE	;GET TYPE OF WRITE
	DCR	A	;IS IT TO THE DIRECTORY
	JRZ	WRITIT	;IF SO WRITE IT OUT

	JMPR	GOODOP	;IF NOT SHOW A SUCCESSFUL R/W OPERATION	


;CLEAR HOST BUFFER FOR DIRECTORY WRITE

WRITIT:

	STA	HSTWRT
	CALL	WRTHST
GOODOP:	XRA	A
	RET		;SUCCESSFUL READ/WRITE



;WRITE FROM THE HOST BUFFER

FLUSH:
WRTHST:
	XRA	A
	RET


RDHST:
	LDA	HSTDSK
	STA	PDISK
	LXI	H,RDBUFF ;POINT TO READ BUFFER
	SHLD	PDMA	;MAKE IT PHYSICAL DMA ADDRESS
	LDA	CTRACK
	STA	PTRACK
	LDA	CSECT
	STA	PSECT

	CALL	PREAD	;READ SECTOR INTO READ BUFFER
	RNC		;RETURN IF READ WAS SUCCESSFUL

	MVI	C,ERR3	;PRINT ERROR
	JMP	EPRINT


;GET ACTUAL TRACK TO SEEK

GETTRK:
	LDA	SEKTRK	;GET CP/M TRACK NUMBER
	LXI	H,OFF	;GET NUMBER OF SYSTEM TRACKS
	MOV	E,M
	MVI	C,0	;ASSUME SINGLE SIDED DISK
	LXI	H,SSLEN	;POINT TO SYSTEM SECTOR LENGTH
	CMP	E
	JRC	..SYST	;IT WAS A SYSTEM TRACK
	LXI	H,FLAG	;POINT TO FLAG BYTE
	BIT	1,M	;TEST SIDES BIT
	JRZ	..SING	;SINGLE SIDED DISK
	ADD	E	;ADD IN OFFSET
	SRLR	A	;COMPUTE PHYSICAL TRACK NUMBER
	RARR	C	;GET SIDE NUMBER BIT
..SING:	LXI	H,USLEN
..SYST:
	STA	CTRACK	;SAVE ACTUAL TRACK NUMBER
		

;GET ACTUAL SECTOR TO READ/WRITE

PYSEC:
	MOV	A,M	;GET LENGTH BYTE
	MOV	B,A
	CPI	3
	JRNZ	..LRG
	MVI	H,7
	JMPR	..GSEC
..LRG:	CPI	2	;512 BYTE SECTOR(y/n)
	JRNZ	..NO	;NOPE, ACC HAS RECORD MASK
	INR	A	;FIND MASK FOR 512 BYTE SECTOR
..NO:	MOV	H,A	;SAVE RECORD MASK

..GSEC:	LDA	SEKSEC	;GET CP/M SECTOR NUMBER
	DCR	A	;ADJUST DOWN TO START AT ZERO
	MOV	L,A	;SAVE FOR LATER
	INR	B	;ADJUST FOR EASY LOOP
	JMPR	..JOIN
..LOOP:	SRLR	A	;PLACE SECTOR NUMBER IN LSB'S
..JOIN:	DJNZ	..LOOP	;REPEAT UNTIL ALIGNED IN LSB'S
	INR	A	;ADJUST TO MAKE PHYSICAL SECTOR
	ORA	C	;GET SIDE BIT	
	STA	SEKHST	;HOST SECTOR
	STA	CSECT	;SAVE COMBINED SECTOR
	MOV	A,L	;GET CP/M SECTOR NUMBER
	ANA	H	;MASK ALL BUT RECORD NUMBER
	STA	CREC	;SAVE RECORD NUMBER
	RET


;****************************************************
; FLOPPY DISK PHYSICAL READ AND WRITE ROUTINE. ALL
; FLOPPY DISK I/O IS PERFORMED BY CALLS TO THIS
; ROUTINE. ON ENTRY PDISK, PTRACK, PSECT, AND PDMA
; MUST BE VALID.
;****************************************************

PWRITE:	MVI	A,WWRITE ;SET WRITE COMMAND
	JMPR	PCOM..	;JOIN COMMON CODE

PREAD:	MVI	A,WREAD	;SET READ COMMAND

PCOM..:	STA	PCMD	;REMEMBER WHETHER READ OR WRITE
	CALL	GETAPB	;MAKE SURE ATABLE IS RIGHT ONE


;******************************************************	
;	IF LAST I/O WAS ON DIFFERENT DISK, TELL THE
;	179X TO UNLOAD ITS HEAD. THE HEAD LOAD ONE-SHOT
;	WILL THEN BE RETRIGERED ON THE NEXT COMMAND.
;******************************************************

	LXI	H,OLDFLO ;POINT TO OLD FLOPPY NUMBER
	LDA	PDISK	;GET NEW NUMBER
	CMP	M	;SAME DISK(y/n)
	JRZ	..SADS	;YES

	MOV	M,A	;NO, UPDATE OLD NUMBER TO NEW
	CALL	TRIMWT	;WAIT FOR TRIM ERASE TO END
	CALL	SETD3S	;SET Z3S CONTROL BYTE
	CALL	FUNLD	;UNLOAD HEAD
	CALL	FDONE	;INSURE FDC IS DONE


;****************************************************
;	INITIALIZE RETRY LIMITS. INSURE Z3S BYTE IS
;	SET. SEEK TO CORRECT TRACK.
;****************************************************

..SADS:	LDA	FMACRO	;INITIALIZE MACRO RETRY LIMIT
	STA	RMACRO
	MVI	A,3	;RESET MACRO RETRY MAJOR LOOP
	STA	MRML
..MAC:	LDA	FMICRO	;INITIALIZE MICRO RETRY LIMIT
	STA	RMICRO
	CALL	SETD3S	;SET Z3S CONTROL BYTE
	LXI	H,TRACK ;GET OLD TRACK NUMBER
	MOV	A,M
	OUT	WTRACK	;UPDATE 179X TRACK REG
	LDA	PTRACK	;GET DESIRED TRACK NUMBER
	CMP	M	;SAME AS BEFORE(y/n)
	JRZ	..SATR	;YES

	MOV	M,A	;UPDATE TRACK NUMBER
	CALL	TRIMWT	;WAIT FOR TRIM ERASE TO END
	LDA	PTRACK	;GET DESIRED TRACK
	ORA	A	;TRACK 0 DESIRED(y/n)
	JRNZ	..NOZE	;NOPE

	CALL	FHOME	;SEEK TO TRACK 0 BY HOME CMD
	RC
	JMPR	..ENDS	;DONE SEEKING

..NOZE:	CALL	FSEEK	;NORMAL SEEK TO DESIRED TRACK
	RC

..ENDS:	CALL	PUTAPB	;UPDATE APB FROM ATABLE
	LDA	PCMD	;GET READ/WRITE COMMAND
	SET	WBDEL,A	;INSURE HEAD IS SETTLED
	STA	PCMD	;   BY SETTING DELAY BIT IN CMD


;****************************************************
;	SET UP DMA ADDRESS, SECTOR REGISTER. ISSUE
;	THE READ OR WRITE COMMAND. SET HARDWARE WAIT.
;****************************************************

..SATR:
..MIC:
	CALL	DISINT	;DISABLE INTERRUPTS
	LDA	PSECT	;GET DESIRED SECTOR NUMBER
	MOV	B,A	;SAVE SIDE BIT
	ANI	07FH	;DROP SIDE BIT
	OUT	WSECT	;UPDATE 179X SECTOR REGISTER
	LDA	PCMD	;GET READ OR WRITE COMMAND
	BIT	7,B	;ARE WE ON SIDE 1(y/n)
	JRZ	..SID0	;NO, LEAVE SSO BIT AS 0
	SET	WBSID1,A;YES,UPDATE SSO BIT TO 1
..SID0:
	OUT	WCMD	;FOR PIO 179X COMMAND
..SID2:	STA	OCMD
	STA	ORWCMD	;SAVE LAST READ OR WRITE CMD
	MOV	D,A	;SAVE THE COMMAND

	LDA	D3SWT	;GET WAIT ACTIVE Z3S BYTE
	OUT     DCMD	

	STA	OD3S

	LXI	B,128<8+WDATA ;SET PORT AND LENGTH
	LDA	PTRACK	;GET CURRENT TRACK
	LXI	H,OFF
	CMP	M	;IS IT A USER TRACK(y/n)
	LDA	USLEN	;GET USER SECTOR LENGTH
	JRNC	..ULEN
	LDA	SSLEN	;GET SYSTEM SECTOR LENGTH

..ULEN:	LHLD	PDMA	;GET DMA ADDRESS


;****************************************************
;	         READ THE SECTOR.
;****************************************************

..PIRD:
	ORA	A
	JRZ	..PR12
	MVI	B,0H
	DCR	A
	JRZ	..PR25
	DCR	A
	JRZ	..PR51
	INIR
	INIR
..PR51:	INIR
..PR25:
..PR12:	INIR
	
..RDFN:

	MVI	C,WSREAD ;STATUS BITS TO TEST

;****************************************************
;	WAIT FOR COMPLETION OF DISK OPERATION.
;	TEST FOR ERRORS.
;****************************************************

..CHEK:	
	LDA	D3SNO	;GET NO WAIT Z3S BYTE
	OUT     DCMD
	STA	OD3S
	CALL	FQDONE	;WAIT FOR 179X DONE
	ANA	C	;ANY ERROR BITS(y/n)
	RZ		;NO, RETURN - SUCCESSFUL


;****************************************************
;	RETRY THE I/O IF AN ERROR OCCURED.
;****************************************************

	LXI	H,RMACRO ;POINT TO MACRO RETRY COUNT
	DCR	M	;DECREMENT IT
	STC
	RZ		;GIVE UP IF COUNTED DOWN TO 0

	MOV	B,A	;SAVE ERROR STATUS BITS
	LDA	FSOFT	;GET SOFT ERROR REPORT FLAG
	ORA	A	
	JRZ	..NORE	;DON'T REPORT THE SOFT ERROR
	MOV	A,B
	MVI	C,ERR8
	CALL	EPRINT	;PRINT OUT ERROR
	CALL	MSG
	.ASCIS	\ Soft \
..NORE:	LXI	H,RMICRO ;POINT TO MICRO RETRY COUNT
	DCR	M	;DECREMENT IT
	JNZ	..MIC	;INITIATE MICRO RETRY SEQUENCE
	LXI	H,MRML	;POINT MACRO MAJOR LOOP
	DCR	M	;COUNT DOWN
	JRNZ	..MJRR	;MAJOR RETRY
	
	MVI	A,1
	STA	MRML
	MVI	C,ERR8
	LDA	ERRSUP	;GET ERROR SUPPRESSION BYTE
	ORA	A	;TEST IT
	MOV	A,B
	CZ	EPRINT


;****************************************************
;THE FOLLOWING ROUTINE RINGS BELL & FLASHES LED ON
;DRIVE WITH COUNTED DOWN MICRO RETRY
;****************************************************

	CALL	FHOME
	RC
	XRA	A
	STA	TRACK
..CRIN:
	CALL	..RING	;RING BELL
	CALL	CONST	;SEE IF KEY PRESSED
	JRZ	..CRIN	;IF NOT KEEP RINGING
	CALL	CONIN	;GET CHAR
	CPI	3	;A ^C (y/n)
	JZ	0	;DO A WARM BOOT IF SO
	CPI	4	;A ^D (y/n)
	JRZ	..FXER	;IGNORE ERROR
	CPI	0DH	;SEE IF 'CR'
	JRNZ	..CRIN	;IF NOT RING AGAIN
	LDA	ERRSUP	;GET ERROR SUPPRESSION BYTE
	ORA	A
	JRNZ	..MJRR	;SKIP NEXT ON SUPPRESS
	CALL	..BS	;ELSE BACK UP CURSOR
	CALL	..SP	;NOW 'ERASE' PRINTED ERROR MESSAGE
	CALL	..BS	;AND BACK UP CURSOR AGAIN
..MJRR:	CALL	FHOME	;THEN HOME HEAD AND RETRY DISK I/O
	RC		;UNLESS IT'S FATAL
	XRA	A
	STA	TRACK
	JMP	..MAC	;GO DO MACRO RETRY


;IGNORE DISK I/O ERROR AND CONTINUE WITH STATUS SET O.K.

..FXER:
	XRA	A	;CLEAR ERROR FLAG
	RET


..RING:
	CALL	MSG	;RING THE BELL AND DELAY 1 SECOND	
	.BYTE	87H
	RET


;BACK UP CURSOR 8 PLACES TO GET TO START OF ERROR MESSAGE

..BS:
	
	MVI	C,8	;BACKSPACE CHARACTER IN C
..BS0:	MVI	B,8
..BS1:	CALL	CONOUT
	DCR	B
	JRNZ	..BS1
	RET

;PRINT 8 SPACES ON CONSOLE TO 'ERASE' LAST ERROR MESSAGE
;ENTER BACK SPACE ROUTINE AT BS0: TO EXECUTE 8 SPACES

..SP:
	MVI	C,20H	;SPACE CHARACTER IN C
	JMPR	..BS0


;*******************************************************
; SET Z3S CONTROL BYTE. GET 2 Z3S CONTROL BYTES, ONE
; WITH THE HARDWARE WAIT BIT ACTIVE AND ONE WITHOUT,
; FROM THE 8 POSSIBLE CONTROL BYTES. OUTPUT THE NO WAIT
; BYTE. SAVE BOTH FOR LATER USE.
;*******************************************************

SETD3S:	LDA	PTRACK	;GET DESIRED TRACK
	LXI	H,OFF
	CMP	M	;USER TRACK(y/n)
	LXI	H,US0N	;POINT TO USER TRACK BYTES
	JRNC	..USER
	LXI	H,SS0N	;POINT TO SYSTEM TRACK BYTES

..USER:	LDA	PSECT	;TEST IF ON SIDE 1
	BIT	7,A
	JRZ	..SID0	;ON SIDE 0
	INX	H	;POINT TO SIDE 1 BYTES
	INX	H

..SID0:	MOV	E,M	;GET THE NO WAIT BYTE
	INX	H
	MOV	D,M	;GET THE WAIT BYTE
	SDED	D3SNO	;SAVE BOTH
	MOV	A,E
	OUT     DCMD	;OUTPUT THE NO WAIT BYTE
	STA	OD3S
	RET


;*****************************************************
; GET Z3S CONTROL BYTES. CREATE THE EIGHT Z3S CONTROL
; BYTES IN ATABLE. THESE BYTES SET THE DENSITY BIT FOR
; SYSTEM TRACKS AND USER TRACKS, SELECT SIDE 0 OR SIDE
; 1, AND ENABLE OR DISABLE THE HARDWARE DATA WAIT.
; CREATE THE BYTES BY MERGING THE APPROPRIATE HARDWARE
; CONTROL BITS IN OPRTAB WITH THE CORRECT ADDRESS
; CONTROL BITS IN ADRTAB.
;*****************************************************

GETD3S:	LDA	ADISK	;GET CURRENT DISK NUMBER
	LXI	H,ADRTAB;POINT TO ADDRESS
	MOV	E,A	;   CONTROL BITS TABLE
	MVI	D,0
	DAD	D
	MOV	C,M	;SAVE ADDR BITS FOR Z3S BYTES

	LDA	FLAG	;GET THE FLAG
	ANI	00010100B ;ISOLATE SYSTEM DENSITY, SIZE
	MOV	B,A
	RRC		;ALIGN THE DENSITY BIT
	ORA	B	;COMBINE SIZE AND DENSITY
	ANI	00001100B ;DROP UNALIGNED BITS
	LXI	D,SS0N	;POINT TO BEGINNING OF DEST
	CALL	..GET	;GET SYSTEM TRACK Z3S BYTES

	LDA	FLAG	;GET THE FLAG
	ANI	00001100B ;ISOLATE USER DENSITY, SIZE
			;AND GET USER TRACK Z3S BYTES

..GET:	LXI	H,OPRTAB ;POINT TO CONTROL BITS TABLE
	PUSH	D
	MOV	E,A	;GET CONTROL BITS START ADDRESS
	MVI	D,0
	DAD	D	;START ADDRESS NOW IN HL
	POP	D	;DESTINATION RESTORED TO DE
	MVI	B,4
..LOOP:	MOV	A,M	;GET A BYTE OF CONTROL BITS
	INX	H
	ORA	C	;COMBINE WITH ADDRESS BITS
	STAX	D	;STORE COMBINED BYTE
	INX	D
	DJNZ	..LOOP	;DO ALL BYTES
	RET

;****************************************************
; WAIT FOR TRIM ERASE TO END. WAIT ONLY IF THE LAST
; FLOPPY DISK COMMAND WAS A WRITE. CALLED ONLY IF
; PHYSICAL HEAD MOTION IS NEEDED TO ACCESS THE NEXT
; SECTOR. THE WAIT IS ABOUT 500 USEC AT 4 MHZ. THIS
; ALLOWS TRIM ERASE TO COMPLETE BEFORE THE DRIVE IS
; DESELECTED OR THE HEAD IS MOVED.
;****************************************************

TRIMWT:	LDA	ORWCMD	;GET LAST READ OR WRITE COMMAND
	BIT	WBWRIT,A ;TEST WRITE BIT
	RZ		;IT WAS A READ, DON'T WAIT
	MVI	B,150	;WAIT
	DJNZ	. 
	RET


;*****************************************************
; MULTI-PURPOSE 179X SEEK SUBROUTINE. THE ENTRY
; POINTS ARE:

;	FHOME - RESTORE HEAD TO TRACK 0 POSITION
;	FSEEK - SEEK WITH HEAD LOAD TO DEST. IN ACC

;	FUNLD - SEEK SAME TRACK TO UNLOAD HEAD
;	FLOAD - SEEK SAME TRACK TO LOAD HEAD

; FOR THESE LAST TWO FUNCTIONS THE COMMAND IS STILL IN
; PROGRESS WHEN RETURN IS MADE. THE CALLING PROGRAM
; MUST WAIT FOR THE COMMAND TO COMPLETE.
;*****************************************************

FHOME:	MVI	B,WHOME ;SET UP HOME COMMAND
	JMPR	FH..

FSEEK:	OUT	WDATA	;OUTPUT SEEK DESTINATION
	MVI	B,WSEEK	;SET UP SEEK, DIFFERENT TRACK

FH..:	LDA	SPEED	;GET SEEK SPEED
	ORA	B	;MERGE WITH SEEK COMMAND
	OUT	WCMD	;OUTPUT TO 179X
	STA	OCMD
	CALL	FDONE
	ANI	WSSEEK	;ELIMINATE UNWANTED STATUS BITS


;****************************************************
;    NOW GO IN A WAIT LOOP FOR HEAD SETTLE TIME
;****************************************************

SETL:   PUSH	PSW     ;SAVE REGS
        MVI     A,DLY   ;COUNT 1
SETL0:  MVI     B,0     ;COUNT 2
SETL1:  DCR     B       ;LOOP 1
        JRNZ    SETL1   ;DO NEXT
        DCR     A       ;LOOP 2
        JRNZ    SETL0   ;DO THAT ONE NOW
        POP	PSW     ;GET BACK REGS
	RZ		;RETURN - SUCCESSFUL 
	STC
	RET		;RETURN WITH CARRY SET - ERROR

FUNLD:	MVI	B,WUNLD	;SET UP UNLOAD COMMAND
	JMPR	FU..

FLOAD:	MVI	B,WLOAD	;SET UP LOAD COMMAND

FU..:	IN	WTRACK	;GET CURRENT TRACK
	OUT	WDATA	;OUTPUT SEEK DESTINATION

	MOV	A,B	;OUTPUT SEEK COMMAND
	OUT	WCMD
	RET


;******************************************************
; 179X NOT BUSY SUBROUTINE. WAIT FOR 179X NOT BUSY.
; THEN RETURN THE LAST STATUS READ FROM THE CHIP. THERE
; ARE TWO ENTRY POINTS. FDONE DELAYS A SHORT WHILE TO
; ALLOW THE 179X TO SET ITS BUSY BIT.
; THIS ROUTINE TESTS FOR TYPE 1 AND TYPE 2
; COMMAND COMPLETION. 
;******************************************************

FDONE:

	MVI	B,10	;DELAY
	DJNZ	.


FQDONE:
PIODON:
	IN	WSTAT
	STA	ISTAT
	BIT	WBBUSY,A
	JRNZ	PIODON
	RET	
	


;******************************************************
; FLOPPY DISK ERROR PRINT SUBROUTINE. ANY NON ZERO BITS
; IN THE ACCUMULATOR ARE ERRORS. THE FIRST ONE FOUND IS
; PRINTED OUT.
;******************************************************

EPRINT:

	MOV	D,A	;SAVE ERROR BITS
	MOV	A,C	;STORE CALLING ROUTINE ID
	STA	ACTIVE

	IN	WTRACK	;STORE TRACK REGISTER
	STA	ITRACK
	IN	WSECT	;STORE SECTOR REGISTER
	STA	ISECT

EMSROT:
	CALL	MSG
	.ASCIS	\Err \
	LDA	PDISK
	ADI	'A'
	MOV	C,A
	CALL	BIOOUT
	CALL	MSG
	.ASCIS	\: \

	SET	1,D	;INSURE NON ZERO ERROR BYTE
	MVI	B,0
..FIND:	INR	B
	RALR	D	;FIND FIRST NON ZERO BIT
	JRNC	..FIND
	LXI	H,ERMESS ;POINT TO MESSAGE TABLE
	JMPR	..JOIN
..MORE:	BIT	7,M	;TEST FOR END OF MESSAGE
	INX	H
	JRZ	..MORE	;KEEP LOOKING FOR END, THIS MSG
..JOIN:	DJNZ	..MORE	;KEEP LOOKING FOR CORRECT MSG
	CALL	MSGHL	;PRINT OUT CORRECT MESSAGE
	LDA	ACTIVE	;GET CALLING ROUTINE
	CPI	8
	JRZ	..EEX	;O.K. ENOUGH
	CALL	MSG
	.ASCIS	\ \

	MVI	B,EILEN ;NUMBER OF ERROR INFO BYTES
	LXI	H,EINFO
..LOOP:	CALL	MSG	;SPACE BETWEEN BYTES
	.ASCIS	\ \
	MOV	A,M
	INX	H
	CALL	LBYTE	;PRINT NEXT BYTE
	DJNZ	..LOOP

..EEX:	MVI	A,1	;CP/M ERROR FLAG
	STC		;INTERNAL ERROR FLAG
	RET


;****************************************************
;            PRINT BYTE ON CONSOLE.
;****************************************************

LBYTE:	PUSH	PSW
	RRC
	RRC
	RRC
	RRC
	CALL	..2
	POP	PSW
..2:	ANI	0FH
	ADI	90H
	DAA
	ACI	40H
	DAA
	MOV	C,A
	JMP	BIOOUT


;****************************************************
; MOVE ATABLE INTO THE CORRECT APB. MOVE THE CORRECT
; APB INTO ATABLE.
;****************************************************

PUTAPB:	LDA	ADISK	;GET CURRENT ATABLE NUMBER
	CALL	ASET..	;SET UP FOR MOVE
	LDIR		;COPY ATABLE INTO APB
	RET

GETAPB:	LXI	H,ADISK	;POINT TO ATABLE DRIVE NUMBER
	LDA	PDISK	;GET NEW NUMBER
	CMP	M	;THE SAME(y/n)
	RZ		;YES, ATABLE ALREADY VALID
	MOV	M,A	;UPDATE ATABLE NUMBER
	CALL	ASET..	;SET UP FOR MOVE
	XCHG		;APB ADDRESS NOW SOURCE
	LDIR		;COPY APB INTO ATABLE

;NOW GET ALLOCATION SIZE FOR SELECTED DISK

	LDA	PDISK	;GET DISK #
	MOV	E,A	;MAKE INDEX
	MVI	D,0
	LXI	H,ALOCSZ;POINT TO SAVED BSH TABLE
	DAD	D
	MOV	A,M	;GET BSH VALUE
	LXI	H,ALOREC;POINT TO RECORDS/ALLOCATION SIZE
	SUI	3	;SET FOR INDEX
	MOV	E,A	;MAKE INDEX
	DAD	D
	MOV	A,M	;GET RECORDS/ALLOCATION
	STA	ALOCA	;SAVE # FOR SELECTED DRIVE

	RET

;TABLE FOR # OF RECORDS FOR ALLOCATION SIZES FROM 1K TO 16K

ALOREC:
	.BYTE	8,16,32,64,128


ASET..:	MOV	L,A	;GET DISK NUMBER INTO L
	CALL	GETDPH
	LXI	H,ATABLE
	LXI	B,ALEN
	RET


;****************************************************
; GET DPH ADDRESS AND APB ADDRESS. RETURN THE DPH
; ADDRESS IN HL, THE CORRESPONDING APB ADDRESS IN DE.
; DISK NUMBER MUST BE IN L AT CALL.
;****************************************************

GETDPH:	MVI	H,0
	DAD	H	;GET NUMBER * 2
	MOV	E,L
	MOV	D,H
	DAD	H
	DAD	H
	DAD	H	;GET NUMBER * 16
	DAD	D	;GET NUMBER * 18
	LXI	D,APBBEG
	DAD	D	;NOW WE HAVE ADDR OF APB ADDR
	MOV	E,M	;GET APB ADDRESS INTO DE
	INX	H
	MOV	D,M
	INX	H	;HL NOW HAS DPH ADDRESS
	RET


;****************************************************
;                  FATAL ERROR.
;****************************************************

FATERR:	CALL	EPRINT	;PRINT ERROR

QUIT:	CALL	MSG
	.ASCIS	\ STP\
	JMP	0	;GO WARM BOOT


;*****************************************************
; MESSAGE OUTPUT SUBROUTINE. THERE ARE TWO ENTRY
; POINTS. FOR MSG A MESSAGE FOLLOWS INLINE AFTER THE
; CALL. FOR MSGHL THE MESSAGE ADDRESS IS IN HL. FOR
; BOTH ENTRY POINTS THE MESSAGE IS PRINTED OUT UP TO
; AND INCLUDING A CHARACTER WITH ITS HI ORDER BIT SET.
;*****************************************************

MSG:	XTHL		;GET MESSAGE ADDRESS
	CALL	MSGHL	;PRINT THE MESSAGE
	XTHL		;RESTORE NEW RETURN ADDRESS
	RET

MSGHL:	MOV	C,M	;GET NEXT CHAR OF MESSAGE
	INX	H
	MOV	A,C	;JUST IN CASE OF H. R. W.
	CALL	BIOOUT	;OUTPUT THE CHAR
	ORA	A	;TEST FOR HI BIT SET
	JP	MSGHL	;NOT SET, KEEP GOING
	RET




LISTST:
LIST:
READER:
PUNCH:
	RET

;****************************************************
; IO ROUTINES FOR THE OTHER IOBYTE VECTORED DEVICES
;****************************************************

CONST:
	IN	0
	ANI	2
	RZ
	MVI	A,0FFH
	RET

CONIN:
	CALL	CONST
	ORA	A
	JRZ	CONIN
	IN	1
	ANI	7FH
	RET

	.IFE	VDBCON,[
CONOUT:
	IN	0
	ANI	4
	JRZ	CONOUT
	MOV	A,C
	OUT	1
	RET
		][
CONOUT:
	IN	7DH
	ANI	1
	JRZ	CONOUT
	MOV	A,C
	OUT	7CH
	RET	]

;****************************************************
; FLOPPY DISK ERROR MESSAGES. USED BY THE EPRINT
; ROUTINE TO TELL THE USER WHICH 179X ERROR OCCURED.
;****************************************************

ERMESS:
	.ASCIS	\1\	;NOT READY
	.ASCIS	\2\	;WRITE PROTECTED
	.ASCIS	\3\	;FAULT
	.ASCIS	\4\	;RNF
	.ASCIS	\5\	;CRC ERROR
	.ASCIS	\6\	;LOST R/W DATA
	.ASCIS	\?\	;CAN'T READ FORMAT



;****************************************************
; FLOPPY DISK DPH TABLE. FOR THE CONVENIENCE OF THE
; CBIOS THE APB ADDRESS FOR A DISK PRECEDES THE DPH
; FOR THE DISK.
;****************************************************

APBBEG:
	.WORD	APB0
	.WORD	TRANS0	;LOGICAL TO PHYSICAL XLATE TAB
	.WORD	0	;SCRATCH
	.WORD	0
	.WORD	0
	.WORD	DIRBUF	;DIRECTORY BUFFER
	.WORD	DPB0	;DISK PARAMETER BLOCK
	.WORD	CSV0	;CHECKSUM VECTOR
	.WORD	ALV0	;ALLOCATION VECTOR

	.WORD	APB1
	.WORD	TRANS1
	.WORD	0
	.WORD	0
	.WORD	0
	.WORD	DIRBUF
	.WORD	DPB1
	.WORD	CSV1
	.WORD	ALV1

;******************************************************	
; STANDARD APBS, DPBS AND TRANSLATE TABLES. THESE
; TABLES ARE USED BY SELDSK WHEN IT CANNOT FIND A VALID
; DDB ON THE DISK. THE TABLES ARE FOR STANDARD 8" OR 5"
; SINGLE DENSITY FLOPPY DISKS. THIS ALLOWS PROGRAM
; INTERCHANGE WITH OTHER CP/M BASED SYSTEMS WITHOUT
; REQUIRING A DDB TO BE WRITTEN ON EACH DISK.
;******************************************************



;****************************************************
;             8" FLOPPY DISK TABLES.
;****************************************************

STDDDB:	.BYTE	00000001B ;FLAG
	.WORD	2	;OFF
	.BYTE	0	;SSLEN
	.BYTE	26	;SLRPS
	.BYTE	0	;USLEN
	.BYTE	26	;ULRPS

	.WORD	26	;STANDARD DPB
	.BYTE	3,7,0
	.WORD	242
	.WORD	63
	.BYTE	192,0
	.WORD	16
	.WORD	2

	.BYTE	1,7,13,19,25,5,11,17,23 ;STANDARD
	.BYTE	3,9,15,21,2,8,14,20,26  ;TRANSLATE
	.BYTE	6,12,18,24,4,10,16,22   ;TABLE

FIRSTS:	.BYTE	0FFH
	
;****************************************************
;            5" FLOPPY DISK TABLES.
;****************************************************

.IFE	STEP,[
ALTDDB:	.BYTE	00000101B ;FLAG
	.WORD	3	;OFF
	.BYTE	0	;SSLEN
	.BYTE	18	;SLRPS
	.BYTE	0	;USLEN
	.BYTE	18	;ULRPS

	.WORD	18	;STANDARD DPB
	.BYTE	3,7,0
	.WORD	71
	.WORD	63
	.BYTE	192,0
	.WORD	16
	.WORD	3

	.BYTE	1,5,9,13,17,3,7,11,15 ;STANDARD
	.BYTE	2,6,10,14,18,4,8,12,16 ;XLATE TABLE
]


;****************************************************
; END OF BIOS INSTRUCTIONS AND CONSTANTS. BEGINNING
; OF WORK AREA.
;****************************************************


;****************************************************
; CP/M CALL PARAMETER STORAGE. CP/M SETS THE LOGICAL
; PARAMETERS BY PRELIMINARY CALLS BEFORE CALLING THE
; READ OR WRITE ROUTINES. THE CP/M I/O ROUTINES SET
; AND USE THE REMAINING VARIABLES.
;****************************************************

SEKSEC:	.BLKW	1	;LOGICAL SECTOR TO R/W
DMAADD:	.BLKW	1	;LOGICAL & ACTUAL CP/M DMA ADDR
CREC:	.BLKB	1	;CURRENT RECORD WITHIN SECTOR


;******************************************************
; PHYSICAL DISK I/O PARAMETER STORAGE. THESE PARAMETERS
; MUST BE SET BEFORE CALLING THE FLOPPY DISK PHYSICAL
; READ OR WRITE ROUTINES.
;******************************************************

PDISK:	.BLKB	1	;PHYSICAL DISK FOR NEXT I/O
PTRACK:	.BLKW	1	;PHYSICAL TRACK FOR NEXT I/O
PSECT:	.BLKW	1	;PHYSICAL SECTOR FOR NEXT I/O
PDMA:	.BLKW	1	;PHYSICAL BUFFER ADDR, NEXT I/O

CTRACK:	.BLKW	1
CSECT:	.BLKW	1



;****************************************************
;    GENERAL PURPOSE VARIABLES ARE STORED HERE.
;****************************************************

RETRY:
RMICRO:	.BLKB	1	;MICRO FDC RETRY COUNT
RMACRO:	.BLKB	1	;MACRO FDC RETRY COUNT
ADISK:	.BLKB	1	;DISK NUMBER OF DISK AT ATABLE
OLDFLO:	.BLKB	1	;OLD FLOPPY DRIVE NUMBER
PCMD:	.BLKB	1	;ACTUAL FLOPPY READ/WRITE CMD
ORWCMD:	.BLKB	1	;LAST R/W OUTPUT TO CMD REG
DPHADR:	.BLKW	1	;DPH ADDRESS FOR CURRENT DISK
APBADR:	.BLKW	1	;APB ADDRESS FOR CURRENT DISK
WRTYPE:	.BLKB	1	;TYPE OF WRITE
D3SNO:	.BLKB	1	;CURRENT NO WAIT Z3S BYTE
D3SWT:	.BLKB	1	;CURRENT WAIT Z3S BYTE
SAVADR:	.BLKW	1	;STD OR ALT DDB ADDRESS
MRML:	.BLKB	1	;MACRO RETRY MAJOR LOOP


;*******************************************************
; ERROR INFORMATION STORAGE. VARIOUS ROUTINES USE THIS
; AREA TO STORE KEY VARIABLES RELATING TO THE Z3S
; CONTROLLER AND THE 179X CHIP. IF A PERMANENT DISK
; ERROR OCCURS, THE VARIABLES ARE PRINTED OUT AS AN AID
; IN ERROR DIAGNOSIS.
;*******************************************************

EINFO:
ITRACK:	.BLKB	1	;LAST INPUT FROM TRACK REG
ISECT:	.BLKB	1	;LAST INPUT FROM SECTOR REG
ISTAT:	.BLKB	1	;LAST INPUT FROM STATUS REG
ACTIVE:	.BLKB	1	;SHOWS WHICH ROUTINE HAD ERROR
OD3S:	.BLKB	1	;LAST OUTPUT TO Z3S CTRL BYTE
OCMD:	.BLKB	1	;LAST OUTPUT TO CMD REG
EILEN	==	.-EINFO



;*********************************************************
;*   ALLOCATION VALUES FOR SELECTED DISKS ARE NEXT
;*********************************************************

ALOCSZ:	.BLKB	4	;BSH VALUE FOR 4 DRIVES
ALOCA:	.BLKB	1	;RECORDS/BLOCK FOR SELECTED DRIVE

;UNALLOCATED PARAMETERS FOLLOW


UNACNT:	.BLKB	1	;UNALLOCATED RECORD COUNT

UNADSK:	.BLKB	1	;      "     DISK #	
UNATRK:	.BLKW	1	;      "     TRACK #
UNASEC:	.BLKW	1       ;      "     SECTOR #

NEWSEC:	.BLKW	1	;CP/M SET SECTOR

RSFLAG:	.BLKB	1	;READ SECTOR FLAG
READOP:	.BLKB	1	;READ = 0, WRITE = 1
HSTACT:	.BLKB	1	;0 = HOST NOT ACTIVE
HSTWRT:	.BLKB	1	;HOST WRITTEN FLAG


HSTDSK:	.BLKB	1
HSTTRK:	.BLKW	1
HSTSEC:	.BLKW	1

SEKDSK:	.BLKB	1
SEKTRK:	.BLKW	1
SEKHST:	.BLKW	1

	UNALEN	==	.-UNACNT


;*******************************************************
; APB COPY FOR CURRENT DISK. THE APB CONTAINS SEVERAL
; IMPORTANT BYTES NEEDED TO CONTROL THE ACTIVE DISK
; DRIVE. THE EXACT LENGTH AND ORDERING OF THESE ENTRIES
; IS CRITICAL, SO CHANGES MUST BE MADE WITH CARE. ONLY
; THE FLAG BYTE HAS ANY MEANING FOR A HARD DISK.
;*******************************************************

ATABLE:
SS0N:	.BLKB	1	;SYSTEM TRACK Z3S CONTROL BYTES
SS0W:	.BLKB	1
SS1N:	.BLKB	1
SS1W:	.BLKB	1
US0N:	.BLKB	1	;USER TRACK Z3S CONTROL BYTES
US0W:	.BLKB	1
US1N:	.BLKB	1
US1W:	.BLKB	1
TRACK:	.BLKB	1	;CURRENT TRACK
SPEED:	.BLKB	1	;DRIVE SEEK SPEED
FLAG:	.BLKB	1	;FLAG BYTE
OFF:	.BLKW	1	;NUMBER OF SYSTEM TRACKS
SSLEN:	.BLKB	1	;SECTOR LENGTH, SYSTEM
SLRPS:	.BLKB	1	;RECORDS PER SIDE, SYS TRACKS
USLEN:	.BLKB	1	;SECTOR LENGTH, USER TRACKS
ULRPS:	.BLKB	1	;RECORDS PER SIDE, USER TRACKS
ALEN	==	.-ATABLE ;ATABLE LENGTH


;******************************************************
; FLOPPY DISK APBS, DPBS AND TRANSLATE TABLES. THE APB
; FOR EACH DRIVE IS HERE, FOLLOWED BY THE DPB, FOLLOWED
; BY THE TRANSLATE TABLE. EXTERNAL ROUTINES NEEDING TO
; ACCESS THE APB ASSUME THAT IT IMMEDIATELY PRECEDES
; THE DPB.
;******************************************************

APB0:	.BLKB	ALEN	;AUXILIARY PARAMETER BLOCK
DPB0:	.BLKB	15	;DISK PARAMETER BLOCK

DPBLEN	==	.-DPB0

TRANS0:	.BLKB	88	;TRANSLATE TABLE

TRALEN	==	.-TRANS0
APBDIS	==	.-APB0


APB1:	.BLKB	1
DPB1:	.BLKB	1
TRANS1:	.BLKB	1

;****************************************************
;       SECTOR READ AND WRITE BUFFERS.
;****************************************************

HSTBUF:
RDBUFF:
WRBUFF:
	.BLKB	1024



;****************************************************
; CP/M WORK AREA. USED BY CP/M FOR DIRECTORY
; OPERATIONS, FLOPPY DISK ALLOCATION VECTORS,
; AND FLOPPY DISK CHANGED DISK CHECKSUMS.
;****************************************************

DIRBUF:	.BLKB	128	;DIRECTORY OPERATION BUFFER

ALV0:	.BLKB	108	;ALLOCATION VECTOR
CSV0:	.BLKB	68	;CHECKSUM VECTOR

ALV1:	.BLKB	1
CSV1:	.BLKB	1


	.END
